"""ShackleAI Loaner Agent — runs on a Loaner's PC to store file chunks."""

import argparse
import hashlib
import json
import os
import platform
import sys
import time
import threading
from pathlib import Path

# Force unbuffered output for Windows background processes
sys.stdout.reconfigure(line_buffering=True)
sys.stderr.reconfigure(line_buffering=True)

import requests

from config import SERVER_URL, HEARTBEAT_INTERVAL


class ShackleAgent:
    def __init__(self, server_url: str, storage_dir: Path, instance_name: str):
        self.server_url = server_url.rstrip("/")
        self.storage_dir = Path(storage_dir)
        self.storage_dir.mkdir(parents=True, exist_ok=True)
        self.instance_name = instance_name
        self.token = None
        self.node_id = None
        self.running = False
        self.token_file = self.storage_dir / ".token"

    def _headers(self):
        return {"Authorization": f"Bearer {self.token}"}

    def _api(self, method, path, **kwargs):
        url = f"{self.server_url}{path}"
        resp = getattr(requests, method)(url, headers=self._headers(), **kwargs)
        if resp.status_code >= 400:
            print(f"  [ERROR] {method.upper()} {path} -> {resp.status_code}: {resp.text}")
        return resp

    # --- Registration ---

    def register_or_login(self, email: str, password: str, name: str, city: str, storage_gb: float, port: int):
        """Register user + node, or login if already registered."""
        # Try login first
        resp = requests.post(f"{self.server_url}/api/auth/login", json={"email": email, "password": password})
        if resp.status_code == 200:
            self.token = resp.json()["access_token"]
            print(f"[AGENT] Logged in as {email}")
        else:
            # Register
            resp = requests.post(f"{self.server_url}/api/auth/register", json={
                "email": email, "password": password, "role": "LOANER",
                "name": name, "city": city,
            })
            if resp.status_code != 201:
                print(f"[AGENT] Registration failed: {resp.text}")
                sys.exit(1)
            print(f"[AGENT] Registered user: {email}")

            # Login
            resp = requests.post(f"{self.server_url}/api/auth/login", json={"email": email, "password": password})
            self.token = resp.json()["access_token"]

        # Save token
        self.token_file.write_text(self.token)

        # Check if node already exists
        resp = self._api("get", "/api/nodes/me")
        if resp.status_code == 200:
            self.node_id = resp.json()["id"]
            print(f"[AGENT] Existing node found: {self.node_id[:8]}...")
        else:
            # Register node
            cpu_cores = os.cpu_count() or 1
            ram_gb = round(os.sysconf("SC_PAGE_SIZE") * os.sysconf("SC_PHYS_PAGES") / (1024**3), 1) if hasattr(os, "sysconf") else 8.0
            resp = self._api("post", "/api/nodes/register", json={
                "name": self.instance_name,
                "total_storage_gb": storage_gb,
                "city": city,
                "cpu_cores": cpu_cores,
                "ram_gb": ram_gb,
                "port": port,
            })
            if resp.status_code == 201:
                self.node_id = resp.json()["id"]
                print(f"[AGENT] Node registered: {self.instance_name} ({storage_gb}GB, {city})")
            else:
                print(f"[AGENT] Node registration failed: {resp.text}")
                sys.exit(1)

    # --- Heartbeat ---

    def _heartbeat_loop(self):
        while self.running:
            try:
                used = self._get_used_storage()
                self._api("post", "/api/nodes/heartbeat", json={"used_storage_gb": used})
            except Exception as e:
                print(f"  [HEARTBEAT] Error: {e}")
            time.sleep(HEARTBEAT_INTERVAL)

    def _get_used_storage(self) -> float:
        """Calculate total GB of chunks stored locally."""
        total = sum(f.stat().st_size for f in self.storage_dir.glob("*.chunk"))
        return round(total / (1024**3), 4)

    # --- Chunk Management ---

    def pull_chunks(self):
        """Poll server for assigned chunks and download any we don't have yet."""
        resp = self._api("get", "/api/nodes/me/chunks")
        if resp.status_code != 200:
            return

        chunks = resp.json()
        new_count = 0
        for chunk in chunks:
            chunk_id = chunk["chunk_id"]
            local_path = self.storage_dir / f"{chunk_id}.chunk"

            if local_path.exists():
                continue  # Already have it

            # Download from server
            data_resp = self._api("get", f"/api/nodes/chunks/{chunk_id}/data")
            if data_resp.status_code != 200:
                continue

            data = data_resp.content

            # Verify integrity
            actual_hash = hashlib.sha256(data).hexdigest()
            if actual_hash != chunk["chunk_hash"]:
                print(f"  [CHUNK] INTEGRITY FAIL for {chunk_id[:8]}... (expected {chunk['chunk_hash'][:16]})")
                continue

            # Store locally
            local_path.write_bytes(data)
            new_count += 1

            # Confirm storage
            self._api("post", f"/api/nodes/chunks/{chunk_id}/stored")
            print(f"  [CHUNK] Stored {chunk_id[:8]}... ({chunk['chunk_size_bytes']} bytes, hash OK)")

        if new_count:
            print(f"[AGENT] Pulled {new_count} new chunk(s)")

    def _chunk_poll_loop(self):
        """Periodically check for new chunk assignments."""
        while self.running:
            try:
                self.pull_chunks()
            except Exception as e:
                print(f"  [POLL] Error: {e}")
            time.sleep(10)  # Poll every 10 seconds

    # --- Main Loop ---

    def start(self):
        """Start heartbeat and chunk polling loops."""
        self.running = True
        print(f"\n[AGENT] {self.instance_name} is ONLINE")
        print(f"  Server: {self.server_url}")
        print(f"  Storage: {self.storage_dir}")
        print(f"  Heartbeat: every {HEARTBEAT_INTERVAL}s")
        print(f"  Chunk poll: every 10s")
        print("  Press Ctrl+C to stop.\n")

        hb_thread = threading.Thread(target=self._heartbeat_loop, daemon=True)
        poll_thread = threading.Thread(target=self._chunk_poll_loop, daemon=True)
        hb_thread.start()
        poll_thread.start()

        try:
            while True:
                time.sleep(1)
        except KeyboardInterrupt:
            self.running = False
            print(f"\n[AGENT] {self.instance_name} shutting down.")


def main():
    parser = argparse.ArgumentParser(description="ShackleAI Loaner Agent")
    parser.add_argument("--server", default=SERVER_URL, help="Server URL")
    parser.add_argument("--storage-dir", default=None, help="Local chunk storage directory")
    parser.add_argument("--name", default=None, help="Node display name")
    parser.add_argument("--email", required=True, help="Loaner account email")
    parser.add_argument("--password", default="pass123", help="Account password")
    parser.add_argument("--city", default="Local", help="City for this node")
    parser.add_argument("--storage-gb", type=float, default=50.0, help="Storage to offer (GB)")
    parser.add_argument("--port", type=int, default=9000, help="Agent port (for multi-instance)")
    args = parser.parse_args()

    instance_name = args.name or f"Node-{args.port}"
    storage_dir = Path(args.storage_dir) if args.storage_dir else Path(__file__).parent / f"storage_{args.port}"

    agent = ShackleAgent(
        server_url=args.server,
        storage_dir=storage_dir,
        instance_name=instance_name,
    )

    agent.register_or_login(
        email=args.email,
        password=args.password,
        name=instance_name,
        city=args.city,
        storage_gb=args.storage_gb,
        port=args.port,
    )

    agent.start()


if __name__ == "__main__":
    main()
