"""ShackleAI Loaner Agent — GUI Dashboard with System Tray
Minimizes to tray on close. Agent keeps running in background.
"""

import json
import hashlib
import os
import sys
import time
import threading
import tkinter as tk
from pathlib import Path

try:
    import pystray
    from PIL import Image, ImageDraw
    HAS_TRAY = True
except ImportError:
    HAS_TRAY = False

# ---- Config ----
INSTALL_DIR = Path.home() / "ShackleAgent"
CONFIG_FILE = INSTALL_DIR / "config.json"
HEARTBEAT_INTERVAL = 30
POLL_INTERVAL = 10

# Colors
BG = "#0a0a0f"
BG2 = "#111827"
BG3 = "#1f2937"
GREEN = "#10b981"
GREEN_DIM = "#065f46"
RED = "#ef4444"
RED_DIM = "#7f1d1d"
YELLOW = "#f59e0b"
WHITE = "#ffffff"
GRAY = "#9ca3af"
GRAY_DIM = "#6b7280"


def create_tray_icon_image(color="#10b981"):
    """Create a simple colored circle icon for the system tray."""
    img = Image.new("RGBA", (64, 64), (0, 0, 0, 0))
    draw = ImageDraw.Draw(img)
    # Green circle with dark border
    draw.ellipse([4, 4, 60, 60], fill=color, outline="#065f46", width=2)
    # "S" letter in white
    draw.text((20, 12), "S", fill="white",
              font=None)  # default font, small but visible
    return img


class AgentUI:
    def __init__(self):
        self.root = tk.Tk()
        self.root.title("ShackleAI Agent")
        self.root.geometry("420x620")
        self.root.resizable(False, False)
        self.root.configure(bg=BG)
        self.root.protocol("WM_DELETE_WINDOW", self.minimize_to_tray)

        # Center
        self.root.update_idletasks()
        x = (self.root.winfo_screenwidth() - 420) // 2
        y = (self.root.winfo_screenheight() - 620) // 2
        self.root.geometry(f"420x620+{x}+{y}")

        # State
        self.running = False
        self.token = None
        self.node_id = None
        self.config = {}
        self.chunks_stored = 0
        self.last_heartbeat = "Never"
        self.status = "STARTING"
        self.errors = []
        self.storage_dir = INSTALL_DIR / "storage"
        self.tray_icon = None
        self.window_visible = True

        self.build_ui()
        self.load_config()

    def build_ui(self):
        # Header
        header = tk.Frame(self.root, bg=BG, pady=12)
        header.pack(fill="x")
        tk.Label(header, text="ShackleAI", font=("Segoe UI", 18, "bold"),
                 fg=WHITE, bg=BG).pack()
        tk.Label(header, text="Loaner Agent", font=("Segoe UI", 9),
                 fg=GRAY_DIM, bg=BG).pack()

        # Status indicator (big)
        status_frame = tk.Frame(self.root, bg=BG, pady=8)
        status_frame.pack(fill="x")

        self.status_dot = tk.Canvas(status_frame, width=16, height=16,
                                     bg=BG, highlightthickness=0)
        self.status_dot.pack(side="left", padx=(30, 8))
        self.status_dot.create_oval(2, 2, 14, 14, fill=YELLOW, outline="", tags="dot")

        self.status_label = tk.Label(status_frame, text="Starting...",
                                      font=("Segoe UI", 13, "bold"),
                                      fg=YELLOW, bg=BG)
        self.status_label.pack(side="left")

        # Separator
        tk.Frame(self.root, bg=BG3, height=1).pack(fill="x", padx=20, pady=10)

        # Info grid
        info_frame = tk.Frame(self.root, bg=BG, padx=25)
        info_frame.pack(fill="x")

        self.info_vars = {}
        fields = [
            ("Node Name", "name", "--"),
            ("Server", "server", "--"),
            ("Email", "email", "--"),
            ("Storage Shared", "storage", "--"),
            ("Chunks Stored", "chunks", "0"),
            ("Data Stored", "data", "0 B"),
            ("Last Heartbeat", "heartbeat", "Never"),
            ("Uptime", "uptime", "--"),
        ]
        for label, key, default in fields:
            row = tk.Frame(info_frame, bg=BG)
            row.pack(fill="x", pady=3)
            tk.Label(row, text=label, font=("Segoe UI", 9),
                     fg=GRAY_DIM, bg=BG, width=15, anchor="w").pack(side="left")
            var = tk.StringVar(value=default)
            tk.Label(row, textvariable=var, font=("Segoe UI", 9, "bold"),
                     fg=WHITE, bg=BG, anchor="w").pack(side="left", fill="x")
            self.info_vars[key] = var

        # Separator
        tk.Frame(self.root, bg=BG3, height=1).pack(fill="x", padx=20, pady=10)

        # Log area
        log_label = tk.Label(self.root, text="Activity Log", font=("Segoe UI", 9),
                              fg=GRAY_DIM, bg=BG, anchor="w")
        log_label.pack(fill="x", padx=25)

        log_frame = tk.Frame(self.root, bg=BG2, bd=0, padx=25, pady=3,
                              highlightthickness=0)
        log_frame.pack(fill="both", expand=True, padx=20, pady=(3, 5))

        self.log_text = tk.Text(log_frame, font=("Consolas", 8),
                                 bg=BG2, fg=GRAY, relief="flat", bd=5,
                                 highlightthickness=1, highlightbackground=BG3,
                                 wrap="word", state="disabled", height=6)
        self.log_text.pack(fill="both", expand=True)
        self.log_text.tag_configure("green", foreground=GREEN)
        self.log_text.tag_configure("red", foreground=RED)
        self.log_text.tag_configure("yellow", foreground=YELLOW)

        # Bottom buttons
        btn_frame = tk.Frame(self.root, bg=BG, pady=10, padx=20)
        btn_frame.pack(fill="x")

        self.toggle_btn = tk.Button(btn_frame, text="Stop Agent",
                                     font=("Segoe UI", 10, "bold"),
                                     bg=RED_DIM, fg=WHITE, relief="flat",
                                     activebackground=RED, activeforeground=WHITE,
                                     padx=20, pady=6, cursor="hand2",
                                     command=self.toggle_agent)
        self.toggle_btn.pack(side="left")

        minimize_btn = tk.Button(btn_frame, text="Minimize to Tray",
                                  font=("Segoe UI", 10),
                                  bg=BG3, fg=GRAY, relief="flat",
                                  activebackground="#374151", activeforeground=WHITE,
                                  padx=15, pady=6, cursor="hand2",
                                  command=self.minimize_to_tray)
        minimize_btn.pack(side="left", padx=(10, 0))

        dash_btn = tk.Button(btn_frame, text="Dashboard",
                              font=("Segoe UI", 10),
                              bg=BG3, fg=GRAY, relief="flat",
                              activebackground="#374151", activeforeground=WHITE,
                              padx=15, pady=6, cursor="hand2",
                              command=self.open_dashboard)
        dash_btn.pack(side="right")

        # Tray hint
        if HAS_TRAY:
            tk.Label(self.root, text="Closing this window minimizes to system tray. Agent keeps running.",
                     font=("Segoe UI", 7), fg=GRAY_DIM, bg=BG).pack(pady=(0, 5))

    # ---- System Tray ----

    def minimize_to_tray(self):
        """Hide window and show system tray icon."""
        if not HAS_TRAY:
            # No tray support — just minimize to taskbar
            self.root.iconify()
            return

        self.root.withdraw()
        self.window_visible = False

        if self.tray_icon is None:
            self._create_tray_icon()

    def _create_tray_icon(self):
        """Create system tray icon with menu."""
        icon_image = create_tray_icon_image()

        menu = pystray.Menu(
            pystray.MenuItem("ShackleAI Agent", self.show_window, default=True),
            pystray.Menu.SEPARATOR,
            pystray.MenuItem(
                lambda item: f"Status: {self.status}",
                None,
                enabled=False
            ),
            pystray.MenuItem(
                lambda item: f"Chunks: {self.chunks_stored}",
                None,
                enabled=False
            ),
            pystray.Menu.SEPARATOR,
            pystray.MenuItem("Open Dashboard", self._tray_open_dashboard),
            pystray.MenuItem("Show Window", self.show_window),
            pystray.Menu.SEPARATOR,
            pystray.MenuItem("Quit", self.quit_app),
        )

        self.tray_icon = pystray.Icon("ShackleAI", icon_image,
                                        "ShackleAI Agent — Running", menu)
        threading.Thread(target=self.tray_icon.run, daemon=True).start()

    def show_window(self, *args):
        """Restore window from tray."""
        def _show():
            self.root.deiconify()
            self.root.lift()
            self.root.focus_force()
            self.window_visible = True
        self.root.after(0, _show)

    def _tray_open_dashboard(self, *args):
        self.open_dashboard()

    def quit_app(self, *args):
        """Fully quit — stop agent and exit."""
        self.running = False
        if self.tray_icon:
            self.tray_icon.stop()
            self.tray_icon = None
        self.root.after(0, self.root.destroy)

    # ---- Logging & Status ----

    def log(self, msg, tag=""):
        def _log():
            self.log_text.configure(state="normal")
            timestamp = time.strftime("%H:%M:%S")
            if tag:
                self.log_text.insert("end", f"[{timestamp}] {msg}\n", tag)
            else:
                self.log_text.insert("end", f"[{timestamp}] {msg}\n")
            self.log_text.see("end")
            self.log_text.configure(state="disabled")
        self.root.after(0, _log)

    def set_status(self, status):
        def _update():
            self.status = status
            if status == "ONLINE":
                self.status_label.configure(text="Online — Earning", fg=GREEN)
                self.status_dot.itemconfig("dot", fill=GREEN)
            elif status == "CONNECTING":
                self.status_label.configure(text="Connecting...", fg=YELLOW)
                self.status_dot.itemconfig("dot", fill=YELLOW)
            elif status == "OFFLINE":
                self.status_label.configure(text="Offline — Stopped", fg=GRAY_DIM)
                self.status_dot.itemconfig("dot", fill=GRAY_DIM)
            elif status == "ERROR":
                self.status_label.configure(text="Error", fg=RED)
                self.status_dot.itemconfig("dot", fill=RED)
            else:
                self.status_label.configure(text=status, fg=YELLOW)
                self.status_dot.itemconfig("dot", fill=YELLOW)
            # Update tray tooltip
            if self.tray_icon:
                self.tray_icon.title = f"ShackleAI Agent — {status}"
        self.root.after(0, _update)

    def update_info(self, key, value):
        def _update():
            if key in self.info_vars:
                self.info_vars[key].set(str(value))
        self.root.after(0, _update)

    # ---- Agent Logic ----

    def load_config(self):
        if not CONFIG_FILE.exists():
            self.set_status("ERROR")
            self.log("No config found. Please run the installer first.", "red")
            return

        self.config = json.loads(CONFIG_FILE.read_text())
        self.update_info("name", self.config.get("name", "--"))
        self.update_info("server", self.config.get("server", "--"))
        self.update_info("email", self.config.get("email", "--"))
        self.update_info("storage", f'{self.config.get("storage_gb", 0)} GB')

        self.storage_dir = INSTALL_DIR / "storage"
        self.storage_dir.mkdir(parents=True, exist_ok=True)

        # Auto-start
        self.start_agent()

    def start_agent(self):
        self.running = True
        self.toggle_btn.configure(text="Stop Agent", bg=RED_DIM)
        self.set_status("CONNECTING")
        self.log("Starting agent...", "yellow")
        threading.Thread(target=self._agent_main, daemon=True).start()

    def stop_agent(self):
        self.running = False
        self.set_status("OFFLINE")
        self.toggle_btn.configure(text="Start Agent", bg=GREEN_DIM)
        self.log("Agent stopped.", "yellow")

    def toggle_agent(self):
        if self.running:
            self.stop_agent()
        else:
            self.start_agent()

    def _api(self, method, path, data=None):
        """Make API call using urllib (no external deps needed for UI)."""
        import urllib.request
        import urllib.error

        url = f'{self.config["server"]}{path}'
        headers = {"Content-Type": "application/json"}
        if self.token:
            headers["Authorization"] = f"Bearer {self.token}"

        body = json.dumps(data).encode() if data else None
        req = urllib.request.Request(url, data=body, headers=headers, method=method.upper())

        try:
            with urllib.request.urlopen(req, timeout=15) as resp:
                return resp.status, json.loads(resp.read())
        except urllib.error.HTTPError as e:
            body = e.read().decode()
            try:
                return e.code, json.loads(body)
            except Exception:
                return e.code, {"detail": body}
        except Exception as e:
            return 0, {"detail": str(e)}

    def _agent_main(self):
        """Main agent loop — login, heartbeat, pull chunks."""
        try:
            # Login
            self.log("Logging in...")
            code, data = self._api("POST", "/api/auth/login", {
                "email": self.config["email"],
                "password": self.config["password"],
            })
            if code != 200:
                self.log(f"Login failed: {data.get('detail', 'Unknown error')}", "red")
                self.set_status("ERROR")
                return
            self.token = data["access_token"]
            self.log("Logged in successfully", "green")

            # Check node
            code, data = self._api("GET", "/api/nodes/me")
            if code == 200:
                self.node_id = data["id"]
                self.log(f"Node found: {data['name']}", "green")
            else:
                # Register node
                self.log("Registering node...")
                code, data = self._api("POST", "/api/nodes/register", {
                    "name": self.config["name"],
                    "total_storage_gb": self.config["storage_gb"],
                    "city": self.config["city"],
                    "cpu_cores": os.cpu_count() or 1,
                    "ram_gb": 8.0,
                    "port": 9000,
                })
                if code in (200, 201):
                    self.node_id = data["id"]
                    self.log(f"Node registered: {data['name']}", "green")
                else:
                    self.log(f"Node registration failed: {data}", "red")
                    self.set_status("ERROR")
                    return

            self.set_status("ONLINE")
            self.log("Agent is ONLINE and earning", "green")
            self._start_time = time.time()

            # Main loops
            hb_thread = threading.Thread(target=self._heartbeat_loop, daemon=True)
            poll_thread = threading.Thread(target=self._chunk_poll_loop, daemon=True)
            hb_thread.start()
            poll_thread.start()

            # Keep alive + update uptime
            while self.running:
                elapsed = int(time.time() - self._start_time)
                hours = elapsed // 3600
                minutes = (elapsed % 3600) // 60
                if hours > 0:
                    self.update_info("uptime", f"{hours}h {minutes}m")
                else:
                    self.update_info("uptime", f"{minutes}m {elapsed % 60}s")
                time.sleep(1)

        except Exception as e:
            self.log(f"Agent error: {e}", "red")
            self.set_status("ERROR")

    def _heartbeat_loop(self):
        while self.running:
            try:
                used = self._get_used_storage()
                code, _ = self._api("POST", "/api/nodes/heartbeat", {"used_storage_gb": used})
                if code == 200:
                    ts = time.strftime("%H:%M:%S")
                    self.update_info("heartbeat", ts)
                    self.last_heartbeat = ts
                else:
                    self.log("Heartbeat failed — will retry", "yellow")
            except Exception as e:
                self.log(f"Heartbeat error: {e}", "red")
            time.sleep(HEARTBEAT_INTERVAL)

    def _chunk_poll_loop(self):
        while self.running:
            try:
                self._pull_chunks()
            except Exception as e:
                self.log(f"Chunk poll error: {e}", "red")
            time.sleep(POLL_INTERVAL)

    def _get_used_storage(self):
        total = sum(f.stat().st_size for f in self.storage_dir.glob("*.chunk"))
        return round(total / (1024 ** 3), 4)

    def _format_bytes(self, b):
        if b == 0:
            return "0 B"
        k = 1024
        sizes = ["B", "KB", "MB", "GB"]
        i = 0
        while b >= k and i < len(sizes) - 1:
            b /= k
            i += 1
        return f"{b:.1f} {sizes[i]}"

    def _pull_chunks(self):
        code, chunks = self._api("GET", "/api/nodes/me/chunks")
        if code != 200:
            return

        # Update chunk count
        self.chunks_stored = len(chunks)
        self.update_info("chunks", str(self.chunks_stored))

        total_bytes = sum(c.get("chunk_size_bytes", 0) for c in chunks)
        self.update_info("data", self._format_bytes(total_bytes))

        new_count = 0
        for chunk in chunks:
            if not self.running:
                break
            chunk_id = chunk["chunk_id"]
            local_path = self.storage_dir / f"{chunk_id}.chunk"

            if local_path.exists():
                continue

            # Download chunk as raw bytes
            import urllib.request
            url = f'{self.config["server"]}/api/nodes/chunks/{chunk_id}/data'
            headers = {"Authorization": f"Bearer {self.token}"}
            req = urllib.request.Request(url, headers=headers)
            try:
                with urllib.request.urlopen(req, timeout=30) as resp:
                    raw_data = resp.read()
            except Exception:
                continue

            # Verify integrity
            actual_hash = hashlib.sha256(raw_data).hexdigest()
            if actual_hash != chunk["chunk_hash"]:
                self.log(f"Chunk {chunk_id[:8]}... hash mismatch!", "red")
                continue

            local_path.write_bytes(raw_data)
            new_count += 1

            # Confirm stored
            self._api("POST", f"/api/nodes/chunks/{chunk_id}/stored")
            size_str = self._format_bytes(chunk.get("chunk_size_bytes", 0))
            self.log(f"Stored chunk {chunk_id[:8]}... ({size_str})", "green")

        if new_count > 0:
            self.log(f"Pulled {new_count} new chunk(s)", "green")

    def open_dashboard(self):
        import webbrowser
        webbrowser.open(f'{self.config["server"]}/loaner')

    def run(self):
        self.root.mainloop()


if __name__ == "__main__":
    app = AgentUI()
    app.run()
