"""ShackleAI Loaner Agent — GUI Installer"""

import json
import os
import shutil
import subprocess
import sys
import threading
import tkinter as tk
from tkinter import ttk, messagebox
from pathlib import Path

SERVER = "https://shackle.errakaaram.com"
INSTALL_DIR = Path.home() / "ShackleAgent"

# Colors matching the ShackleAI brand
BG = "#0a0a0f"
BG2 = "#111827"
BG3 = "#1f2937"
GREEN = "#10b981"
GREEN_DARK = "#059669"
WHITE = "#ffffff"
GRAY = "#9ca3af"
GRAY_DIM = "#6b7280"
RED = "#ef4444"


class InstallerApp:
    def __init__(self):
        self.root = tk.Tk()
        self.root.title("ShackleAI — Loaner Setup")
        self.root.geometry("520x680")
        self.root.resizable(False, False)
        self.root.configure(bg=BG)

        # Try to center on screen
        self.root.update_idletasks()
        x = (self.root.winfo_screenwidth() - 520) // 2
        y = (self.root.winfo_screenheight() - 680) // 2
        self.root.geometry(f"520x680+{x}+{y}")

        self.setup_styles()
        self.current_step = 0
        self.build_ui()

    def setup_styles(self):
        style = ttk.Style()
        style.theme_use("clam")
        style.configure("Green.TButton",
                         background=GREEN, foreground=WHITE,
                         font=("Segoe UI", 11, "bold"),
                         padding=(20, 10))
        style.map("Green.TButton",
                   background=[("active", GREEN_DARK), ("disabled", BG3)])
        style.configure("Dark.TButton",
                         background=BG3, foreground=GRAY,
                         font=("Segoe UI", 10),
                         padding=(15, 8))
        style.map("Dark.TButton",
                   background=[("active", "#374151")])
        style.configure("Green.Horizontal.TProgressbar",
                         troughcolor=BG3, background=GREEN,
                         thickness=6)

    def build_ui(self):
        # Header
        header = tk.Frame(self.root, bg=BG, pady=20)
        header.pack(fill="x")

        title = tk.Label(header, text="ShackleAI", font=("Segoe UI", 22, "bold"),
                         fg=WHITE, bg=BG)
        title.pack()
        subtitle = tk.Label(header, text="Loaner Agent Setup",
                            font=("Segoe UI", 11), fg=GRAY, bg=BG)
        subtitle.pack()

        # Steps indicator
        self.steps_frame = tk.Frame(self.root, bg=BG, pady=5)
        self.steps_frame.pack(fill="x", padx=40)
        self.step_labels = []
        steps = ["Your Details", "Install", "Verify"]
        for i, s in enumerate(steps):
            f = tk.Frame(self.steps_frame, bg=BG)
            f.pack(side="left", expand=True)
            dot = tk.Label(f, text=f"{i+1}", font=("Segoe UI", 9, "bold"),
                           fg=BG if i == 0 else GRAY_DIM,
                           bg=GREEN if i == 0 else BG3,
                           width=3, height=1)
            dot.pack()
            lbl = tk.Label(f, text=s, font=("Segoe UI", 8),
                           fg=GREEN if i == 0 else GRAY_DIM, bg=BG)
            lbl.pack(pady=(2, 0))
            self.step_labels.append((dot, lbl))

        # Separator
        tk.Frame(self.root, bg=BG3, height=1).pack(fill="x", padx=30, pady=15)

        # Content area
        self.content = tk.Frame(self.root, bg=BG)
        self.content.pack(fill="both", expand=True, padx=40)

        # Bottom area
        self.bottom = tk.Frame(self.root, bg=BG, pady=15)
        self.bottom.pack(fill="x", padx=40)

        self.show_step_1()

    def clear_content(self):
        for w in self.content.winfo_children():
            w.destroy()
        for w in self.bottom.winfo_children():
            w.destroy()

    def update_steps(self, active):
        for i, (dot, lbl) in enumerate(self.step_labels):
            if i < active:
                dot.configure(fg=WHITE, bg=GREEN)
                lbl.configure(fg=GREEN)
            elif i == active:
                dot.configure(fg=BG, bg=GREEN)
                lbl.configure(fg=GREEN)
            else:
                dot.configure(fg=GRAY_DIM, bg=BG3)
                lbl.configure(fg=GRAY_DIM)

    def make_field(self, parent, label, default="", show=None):
        tk.Label(parent, text=label, font=("Segoe UI", 9),
                 fg=GRAY, bg=BG, anchor="w").pack(fill="x", pady=(8, 2))
        entry = tk.Entry(parent, font=("Segoe UI", 11),
                         bg=BG2, fg=WHITE, insertbackground=WHITE,
                         relief="flat", bd=0, highlightthickness=1,
                         highlightbackground=BG3, highlightcolor=GREEN)
        if show:
            entry.configure(show=show)
        entry.insert(0, default)
        entry.pack(fill="x", ipady=8, ipadx=8)
        return entry

    # ---- Step 1: User Details ----
    def show_step_1(self):
        self.clear_content()
        self.update_steps(0)

        tk.Label(self.content, text="Enter your details to get started",
                 font=("Segoe UI", 10), fg=GRAY, bg=BG).pack(anchor="w", pady=(0, 5))

        self.email_entry = self.make_field(self.content, "Email Address")
        self.pass_entry = self.make_field(self.content, "Password", show="*")
        self.name_entry = self.make_field(self.content, "Name for this PC",
                                          default=os.environ.get("COMPUTERNAME", "My-PC"))
        self.city_entry = self.make_field(self.content, "Your City")

        # Storage slider
        tk.Label(self.content, text="Storage to Share (GB)",
                 font=("Segoe UI", 9), fg=GRAY, bg=BG, anchor="w").pack(fill="x", pady=(8, 2))
        slider_frame = tk.Frame(self.content, bg=BG)
        slider_frame.pack(fill="x")
        self.storage_var = tk.IntVar(value=50)
        self.storage_label = tk.Label(slider_frame, text="50 GB",
                                       font=("Segoe UI", 12, "bold"),
                                       fg=GREEN, bg=BG, width=8)
        self.storage_label.pack(side="right")
        self.storage_scale = tk.Scale(slider_frame, from_=10, to=500,
                                       orient="horizontal", variable=self.storage_var,
                                       bg=BG, fg=GREEN, troughcolor=BG3,
                                       highlightthickness=0, showvalue=False,
                                       sliderrelief="flat", length=300,
                                       command=self._update_storage_label)
        self.storage_scale.pack(side="left", fill="x", expand=True)

        # Next button
        btn = ttk.Button(self.bottom, text="Install & Connect",
                         style="Green.TButton", command=self.start_install)
        btn.pack(fill="x")

        note = tk.Label(self.bottom, text=f"Installs to {INSTALL_DIR}",
                        font=("Segoe UI", 8), fg=GRAY_DIM, bg=BG)
        note.pack(pady=(8, 0))

    def _update_storage_label(self, val):
        self.storage_label.configure(text=f"{int(float(val))} GB")

    # ---- Step 2: Installing ----
    def start_install(self):
        email = self.email_entry.get().strip()
        password = self.pass_entry.get().strip()
        name = self.name_entry.get().strip()
        city = self.city_entry.get().strip()
        storage = self.storage_var.get()

        if not email or "@" not in email:
            messagebox.showerror("Error", "Please enter a valid email address.")
            return
        if not password or len(password) < 4:
            messagebox.showerror("Error", "Password must be at least 4 characters.")
            return
        if not name:
            messagebox.showerror("Error", "Please enter a name for this PC.")
            return
        if not city:
            messagebox.showerror("Error", "Please enter your city.")
            return

        self.config = {
            "server": SERVER,
            "email": email,
            "password": password,
            "name": name,
            "city": city,
            "storage_gb": storage,
        }

        self.clear_content()
        self.update_steps(1)

        tk.Label(self.content, text="Setting up your node...",
                 font=("Segoe UI", 10), fg=GRAY, bg=BG).pack(anchor="w", pady=(0, 15))

        self.progress = ttk.Progressbar(self.content, style="Green.Horizontal.TProgressbar",
                                         mode="determinate", maximum=100)
        self.progress.pack(fill="x", pady=(0, 15))

        self.log_frame = tk.Frame(self.content, bg=BG2, bd=0,
                                   highlightthickness=1, highlightbackground=BG3)
        self.log_frame.pack(fill="both", expand=True)

        self.log_text = tk.Text(self.log_frame, font=("Consolas", 9),
                                 bg=BG2, fg=GRAY, relief="flat", bd=8,
                                 highlightthickness=0, wrap="word",
                                 state="disabled")
        self.log_text.pack(fill="both", expand=True)
        self.log_text.tag_configure("green", foreground=GREEN)
        self.log_text.tag_configure("red", foreground=RED)
        self.log_text.tag_configure("white", foreground=WHITE)

        threading.Thread(target=self.run_install, daemon=True).start()

    def log(self, msg, tag=""):
        self.log_text.configure(state="normal")
        if tag:
            self.log_text.insert("end", msg + "\n", tag)
        else:
            self.log_text.insert("end", msg + "\n")
        self.log_text.see("end")
        self.log_text.configure(state="disabled")

    def set_progress(self, val):
        self.progress["value"] = val

    def run_install(self):
        try:
            self._do_install()
        except Exception as e:
            self.root.after(0, lambda: self.log(f"ERROR: {e}", "red"))
            self.root.after(0, lambda: self.show_failure(str(e)))

    def _do_install(self):
        # Step 1: Create directory
        self.root.after(0, lambda: self.log("Creating install directory...", "white"))
        INSTALL_DIR.mkdir(parents=True, exist_ok=True)
        self.root.after(0, lambda: self.set_progress(10))
        self.root.after(0, lambda: self.log(f"  -> {INSTALL_DIR}", "green"))

        # Step 2: Copy agent files
        self.root.after(0, lambda: self.log("Copying agent files...", "white"))
        src_dir = Path(__file__).parent / "agent"
        for f in ["agent.py", "config.py", "requirements.txt"]:
            src = src_dir / f
            if src.exists():
                shutil.copy2(src, INSTALL_DIR / f)
                self.root.after(0, lambda fn=f: self.log(f"  -> {fn}", "green"))
        self.root.after(0, lambda: self.set_progress(25))

        # Step 3: Install dependencies
        self.root.after(0, lambda: self.log("Installing dependencies...", "white"))
        result = subprocess.run(
            [sys.executable, "-m", "pip", "install", "-q",
             "-r", str(INSTALL_DIR / "requirements.txt"),
             "pystray", "pillow"],
            capture_output=True, text=True, timeout=180
        )
        if result.returncode != 0:
            self.root.after(0, lambda: self.log(f"  pip error: {result.stderr[:200]}", "red"))
        else:
            self.root.after(0, lambda: self.log("  -> All dependencies installed", "green"))
        self.root.after(0, lambda: self.set_progress(45))

        # Step 4: Save config
        self.root.after(0, lambda: self.log("Saving configuration...", "white"))
        config_path = INSTALL_DIR / "config.json"
        config_path.write_text(json.dumps(self.config, indent=2))
        self.root.after(0, lambda: self.log("  -> config.json saved", "green"))
        self.root.after(0, lambda: self.set_progress(55))

        # Step 5: Test server connection
        self.root.after(0, lambda: self.log("Testing server connection...", "white"))
        import urllib.request
        try:
            req = urllib.request.Request(f"{SERVER}/api/health")
            with urllib.request.urlopen(req, timeout=10) as resp:
                data = json.loads(resp.read())
                self.root.after(0, lambda: self.log(f"  -> Server: {data['status']}", "green"))
        except Exception as e:
            self.root.after(0, lambda: self.log(f"  -> Connection failed: {e}", "red"))
            raise Exception("Cannot reach ShackleAI server. Check your internet connection.")
        self.root.after(0, lambda: self.set_progress(65))

        # Step 6: Register / login
        self.root.after(0, lambda: self.log("Registering account...", "white"))
        import urllib.request
        import urllib.error

        # Try login first
        login_data = json.dumps({"email": self.config["email"],
                                  "password": self.config["password"]}).encode()
        try:
            req = urllib.request.Request(f"{SERVER}/api/auth/login",
                                         data=login_data,
                                         headers={"Content-Type": "application/json"})
            with urllib.request.urlopen(req, timeout=10) as resp:
                token_data = json.loads(resp.read())
                token = token_data["access_token"]
                self.root.after(0, lambda: self.log("  -> Logged in (existing account)", "green"))
        except urllib.error.HTTPError:
            # Register
            reg_data = json.dumps({
                "email": self.config["email"],
                "password": self.config["password"],
                "role": "LOANER",
                "name": self.config["name"],
                "city": self.config["city"],
            }).encode()
            try:
                req = urllib.request.Request(f"{SERVER}/api/auth/register",
                                             data=reg_data,
                                             headers={"Content-Type": "application/json"})
                with urllib.request.urlopen(req, timeout=10) as resp:
                    pass
                self.root.after(0, lambda: self.log("  -> Account created", "green"))

                # Login
                req = urllib.request.Request(f"{SERVER}/api/auth/login",
                                             data=login_data,
                                             headers={"Content-Type": "application/json"})
                with urllib.request.urlopen(req, timeout=10) as resp:
                    token_data = json.loads(resp.read())
                    token = token_data["access_token"]
            except Exception as e:
                raise Exception(f"Registration failed: {e}")
        self.root.after(0, lambda: self.set_progress(80))

        # Step 7: Register node
        self.root.after(0, lambda: self.log("Registering your node...", "white"))
        node_data = json.dumps({
            "name": self.config["name"],
            "total_storage_gb": self.config["storage_gb"],
            "city": self.config["city"],
            "cpu_cores": os.cpu_count() or 1,
            "ram_gb": 8.0,
            "port": 9000,
        }).encode()
        try:
            req = urllib.request.Request(f"{SERVER}/api/nodes/register",
                                         data=node_data,
                                         headers={"Content-Type": "application/json",
                                                   "Authorization": f"Bearer {token}"})
            with urllib.request.urlopen(req, timeout=10) as resp:
                node_info = json.loads(resp.read())
                self.root.after(0, lambda: self.log(f"  -> Node registered: {node_info['name']}", "green"))
        except urllib.error.HTTPError as e:
            if e.code == 409 or e.code == 400:
                self.root.after(0, lambda: self.log("  -> Node already registered", "green"))
            else:
                body = e.read().decode()
                self.root.after(0, lambda: self.log(f"  -> Note: {body[:100]}", "green"))
        self.root.after(0, lambda: self.set_progress(90))

        # Step 8: Create start script
        self.root.after(0, lambda: self.log("Creating launcher...", "white"))

        # Copy the agent UI and uninstaller
        for fname in ["agent_ui.pyw", "uninstall.pyw"]:
            src = Path(__file__).parent / fname
            if src.exists():
                shutil.copy2(src, INSTALL_DIR / fname)

        # Find python.exe (not pythonw.exe)
        python_exe = sys.executable
        if python_exe.endswith("pythonw.exe"):
            python_exe = python_exe.replace("pythonw.exe", "pythonw.exe")
        pythonw_exe = python_exe.replace("python.exe", "pythonw.exe")
        if not Path(pythonw_exe).exists():
            pythonw_exe = python_exe

        # Create launcher bat that opens the GUI agent
        start_bat = INSTALL_DIR / "Start-ShackleAgent.bat"
        start_bat.write_text(
            f'@echo off\n'
            f'start "" "{pythonw_exe}" "{INSTALL_DIR / "agent_ui.pyw"}"\n'
        )

        # Desktop shortcut
        try:
            desktop = Path.home() / "Desktop"
            if desktop.exists():
                shortcut_bat = desktop / "ShackleAI Agent.bat"
                shortcut_bat.write_text(
                    f'@echo off\n'
                    f'start "" "{pythonw_exe}" "{INSTALL_DIR / "agent_ui.pyw"}"\n'
                )
                self.root.after(0, lambda: self.log("  -> Desktop shortcut created", "green"))
        except Exception:
            pass

        self.root.after(0, lambda: self.set_progress(100))
        self.root.after(0, lambda: self.log(""))
        self.root.after(0, lambda: self.log("Installation complete!", "green"))

        # Show success
        self.root.after(500, self.show_success)

    # ---- Step 3: Success ----
    def show_success(self):
        self.clear_content()
        self.update_steps(2)

        # Big checkmark
        tk.Label(self.content, text="✓", font=("Segoe UI", 48),
                 fg=GREEN, bg=BG).pack(pady=(10, 5))
        tk.Label(self.content, text="You're all set!",
                 font=("Segoe UI", 16, "bold"), fg=WHITE, bg=BG).pack()
        tk.Label(self.content, text="Your node is registered and ready to earn.",
                 font=("Segoe UI", 10), fg=GRAY, bg=BG).pack(pady=(5, 20))

        # Info cards
        info_frame = tk.Frame(self.content, bg=BG2, bd=0,
                               highlightthickness=1, highlightbackground=BG3)
        info_frame.pack(fill="x", pady=(0, 10))
        inner = tk.Frame(info_frame, bg=BG2, padx=15, pady=12)
        inner.pack(fill="x")

        for label, value in [
            ("Node Name", self.config["name"]),
            ("Storage", f'{self.config["storage_gb"]} GB'),
            ("Email", self.config["email"]),
            ("Installed To", str(INSTALL_DIR)),
        ]:
            row = tk.Frame(inner, bg=BG2)
            row.pack(fill="x", pady=2)
            tk.Label(row, text=label, font=("Segoe UI", 9),
                     fg=GRAY_DIM, bg=BG2, width=12, anchor="w").pack(side="left")
            tk.Label(row, text=value, font=("Segoe UI", 9),
                     fg=WHITE, bg=BG2, anchor="w").pack(side="left")

        # Buttons
        btn_frame = tk.Frame(self.bottom, bg=BG)
        btn_frame.pack(fill="x")

        start_btn = ttk.Button(btn_frame, text="Start Agent Now",
                                style="Green.TButton",
                                command=self.launch_agent)
        start_btn.pack(fill="x", pady=(0, 8))

        dash_btn = ttk.Button(btn_frame, text="Open Dashboard",
                               style="Dark.TButton",
                               command=lambda: os.startfile(f"{SERVER}/loaner"))
        dash_btn.pack(fill="x")

    def show_failure(self, error):
        self.clear_content()

        tk.Label(self.content, text="✗", font=("Segoe UI", 48),
                 fg=RED, bg=BG).pack(pady=(20, 5))
        tk.Label(self.content, text="Setup Failed",
                 font=("Segoe UI", 16, "bold"), fg=WHITE, bg=BG).pack()
        tk.Label(self.content, text=error,
                 font=("Segoe UI", 10), fg=GRAY, bg=BG, wraplength=400).pack(pady=(10, 20))

        ttk.Button(self.bottom, text="Try Again",
                    style="Green.TButton",
                    command=self.show_step_1).pack(fill="x")

    def launch_agent(self):
        agent_ui = INSTALL_DIR / "agent_ui.pyw"
        if agent_ui.exists():
            pythonw = sys.executable
            if not pythonw.endswith("pythonw.exe"):
                alt = pythonw.replace("python.exe", "pythonw.exe")
                if Path(alt).exists():
                    pythonw = alt
            subprocess.Popen([pythonw, str(agent_ui)])
        else:
            start_bat = INSTALL_DIR / "Start-ShackleAgent.bat"
            if start_bat.exists():
                os.startfile(str(start_bat))
        self.root.after(500, self.root.destroy)

    def run(self):
        self.root.mainloop()


if __name__ == "__main__":
    app = InstallerApp()
    app.run()
